using UnityEngine;
using System.Collections;

public class BGTransitions : MonoBehaviour {
	
	enum ParentDir {TOP, BOTTOM, LEFT, RIGHT, BG, FG};
	
	public Transform prefabQuad = null;
	public Transform topWall = null;
	public Transform leftWall = null;
	public Transform rightWall = null;
	public Transform bottomWall = null;
	public Transform bgPlane = null;
	public Transform fgPlane = null;
	
	public Material[] topMaterials = new Material[7];
	public Material[] bottomMaterials = new Material[7];
	public Material[] leftMaterials = new Material[7];
	public Material[] rightMaterials = new Material[7];
	public Material[] backgrounds = new Material[7];
	public Material[] foregrounds = new Material[7];
	
	private Material[][] materials;
	private Transform[] walls;
	
	private Transform[] lastChildren;
	private Transform[] currentChildren;
	
	private const int NUM_LAYERS = 6;
	private const float EPSILON = 0.00001f;
	
	public void Start() {
		materials = new []{ topMaterials, bottomMaterials, leftMaterials, rightMaterials, backgrounds, foregrounds };
		walls = new[] {topWall, bottomWall, leftWall, rightWall, bgPlane, fgPlane};
		currentChildren = new Transform[NUM_LAYERS];
		SetWalls(Global.currentRound);
	}
	
	
	public void Update() {
		if(Global.isTransitioning)
		{
			float alpha = Mathf.Lerp(0, 1, Global.transitionLerpT);
			
			for(int i=0; i<NUM_LAYERS; i++)
			{
				if(currentChildren[i] != null)
				{
					Color existing = currentChildren[i].renderer.material.color;
					Color newColor = new Color(existing.r, existing.g, existing.b, alpha);
					currentChildren[i].renderer.material.color = newColor;
				}
				
				if(lastChildren[i] != null)
				{
					Color existing = lastChildren[i].renderer.material.color;
					Color newColor = new Color(existing.r, existing.g, existing.b, 1.0f - alpha);
					lastChildren[i].renderer.material.color = newColor;
				}
			}
			
			if(alpha > 1.0 - EPSILON)
			{
				for(int i=0; i<NUM_LAYERS; i++)
				{
					if(currentChildren[i] != null && i < 4)
					{
						currentChildren[i].localPosition = -Vector3.forward * 1.2f;
					}
					if(lastChildren != null && lastChildren[i] != null)
					{
						Destroy(lastChildren[i].gameObject);
					}
				}
				
				lastChildren = null;
			}
		}
	}
	
	
	public void Transition() {
		
		// Make the box go away
		for(int i=0; i<NUM_LAYERS; i++)
		{
			if(i < 4)
				walls[i].renderer.enabled = false;
		}
		
		Global.currentRound++;
		
		for(int i=0; i<NUM_LAYERS; i++)
		{
			if(lastChildren != null && lastChildren[i] != null)
			{
				lastChildren[i].renderer.enabled = false;
			}
		}
		
		lastChildren = currentChildren;
		currentChildren = new Transform[NUM_LAYERS];
		SetWalls(Global.currentRound);
	}
	
	private void SetWalls(int currentRound) {
		
		for(int i = 0; i < NUM_LAYERS; i++) {
			if(i != 5 || currentRound >= 4) {
				Transform child = MakeQuad(currentRound, i);
				child.parent = walls[i];
				currentChildren[i] = child;
				
				if(i < 4)
					child.localScale = Vector3.one;
				else
				{
					child.localScale = new Vector3(9.0f, 9.0f, 6.75f);
					child.localPosition = Vector3.up * currentRound * 0.2f;
				}
			}
			if(currentRound >= 4 && i < 4)
			{
				currentChildren[i].renderer.enabled = false;
			}
		}
	}
	
	private Transform MakeQuad(int offset, int idx) {
		Material newMaterial = materials[idx][offset];
		Vector3 localZero = walls[idx].TransformPoint(-Vector3.forward * 1.5f);
		
		Quaternion quatDir = new Quaternion();
		quatDir.SetLookRotation(Vector3.up);
		
		Transform child = (Transform)Instantiate(prefabQuad, localZero, quatDir);
		child.renderer.material = newMaterial;
		
		Color existing = child.renderer.material.color;	
		Color newColor = new Color(existing.r, existing.g, existing.b, 0.0f);		
		child.renderer.material.color = newColor;
		
		return child;
	}
}
